/*
 * projekat: Seminarski rad iz numerickih metoda
 * naziv   : Iterativni metod za resavanje sistema linearnih jednacina
 * student : Igor Jeremic ( 99/115 ) - igor@jWork.net
 * profesor: Desanka Radunovic
 * asistent: Filip Maric
 * datum   : novembar 2003.
 * dokument: matrica.cc
 * opis    : Definicija apstraktne klase matrica
 *           klasa matrica obezbedjuje osnovne metode za pristup elementima 
 *           matrice, metode za manipulaciju redovima i kolonama matrice,
 *           metode za svodjenje na gornju/donju/diagonalnu trougaonu matricu
 *           kao i metode potrebne za resavanje sistema jednacina 
 *           gausovim direktnim i jacobievim iterativnim postupkom
 *
 *           klasa ne obezbedjuje fizicko cuvanje podataka, za to se brinu
 *           klase koje su izvedene iz klase matrica
**/

#include <iostream>
#include <vector>

#include "matrica.hh"
#include "common.h"
#include "timetest.hh"

using namespace std;

namespace jwork {

#if defined BROJANJE
  unsigned long matrica::broj_get_operacija = 0;
  unsigned long matrica::broj_set_operacija = 0;
  unsigned long matrica::broj_sabiranja = 0;
  unsigned long matrica::broj_mnozenja = 0;
#endif

matrica::matrica(unsigned sirina, unsigned visina) 
  : _sirina(sirina), _visina(visina) { }

void matrica::promeni_dimenzije(unsigned sirina, unsigned visina) {
  _sirina=sirina;
  _visina=visina;
}

void matrica::randomN(T max, T min) {
  for (unsigned j=0;j<visina();j++) 
    for (unsigned i=0;i<sirina();i++) {
      T rnd=T(1+(int) (T(max)*rand()/(RAND_MAX+1.0)));
      if (abs(rnd)>=min)
        set(i,j,rnd);
    }
}

void matrica::napraviJedinicnu() {
  if (!kvadratna())
    throw MATRICA_MORA_BITI_KVADRATNA;
  nule();
  for (unsigned j=0;j<_visina;j++) 
    set(j,j,1);    
}

bool matrica::kvadratna() const {
  return _sirina==_visina;
}

void matrica::zameniKolone(unsigned k1, unsigned k2) {
  if (k1>=sirina() || k2>=sirina())
    throw LOS_INDEKS_MATRICE;
  if (k1!=k2) {
    for (unsigned i=0;i<visina(); i++) {
      T temp=get(k1,i);
      set(k1,i,get(k2,i));
      set(k2,i,temp);
    }
#if defined DEBUG
    cerr << ">> zamena kolona - menjam(" << k1 << ',' << k2 << ")" << endl;
#endif
  }
}

void matrica::zameniRedove(unsigned r1, unsigned r2) {
  if (r1>=visina() || r2>=visina())
    throw LOS_INDEKS_MATRICE;

  if (r1!=r2) {
    for (unsigned i=0;i<sirina(); i++) {
      T temp=get(i,r1);
      set(i,r1,get(i,r2));
      set(i,r2,temp);
    }
#if defined DEBUG
    cerr << ">> zamena redova - menjam(" << r1 << ',' << r2 << ")" << endl;
#endif
  }
}

void matrica::pomnoziIDodajKolonu(unsigned k1, T faktor, unsigned k2) {
  if (k1>=sirina() || k2>=sirina())
    throw LOS_INDEKS_MATRICE;
  for (unsigned i=0;i<visina(); i++) 
    set(k2,i,get(k2,i)+get(k1,i)*faktor);
#if defined DEBUG
    cerr << ">> kolona(" << k2 << ")=kolona(" << k2 << ")+kolona("<< k1 <<")*" << faktor << " " << endl;
#endif
}

void matrica::pomnoziKolonu(unsigned k, T faktor) {
  if (k>=sirina() )
    throw LOS_INDEKS_MATRICE;
  for (unsigned i=0;i<visina(); i++) 
    set(k,i,get(k,i)*faktor);
#if defined DEBUG
    cerr << ">> kolona(" << k << ")=kolona(" << k << ")*" << faktor << " " << endl;
#endif
}

void matrica::pomnoziIDodajRed(unsigned r1, T faktor, unsigned r2) {
  if (r1>=visina() || r2>=visina())
    throw LOS_INDEKS_MATRICE;
  for (unsigned i=0;i<sirina(); i++) 
    set(i,r2,get(i,r2)+get(i,r1)*faktor);
#if defined DEBUG
    cerr << ">> red(" << r2 << ")=red(" << r2 << ")+red("<< r1 <<")*" << faktor << " " << endl;
#endif
}

void matrica::pomnoziRed(unsigned r, T faktor) {
  if (r>=visina())
    throw LOS_INDEKS_MATRICE;
  for (unsigned i=0;i<sirina(); i++) 
    set(i,r,get(i,r)*faktor);
#if defined DEBUG
    cerr << ">> red(" << r << ")=red(" << r << ")*" << faktor << " " << endl;
#endif
}

/*
 * funkcije za ispis (ispisi, i ispisi1) imaju drugi opcioni argument 
 * a to je pokazivac na matricu b, koji je po default-u 0, ali ukoliko
 * se on navede, paralelno sa ispisom matrice bice i ispisana matrica b koja
 * najcesce predstavlja vektor slobodnih clanova u sistemu linearnih jednacina
 */

/* ispis daje izlaz koji je kompatibilan sa programom matlab */

ostream & matrica::ispisi(ostream & ostr, matrica * b) const {

  if (b!=0 && b->visina()!=visina())
    throw VISINE_MORAJU_BITI_JEDNAKE;

  if (b==0) 
    ostr << "(" << sirina() << " x " << visina() << ")" << endl;
  else
    ostr << "(" << sirina() << " x " << visina() << " | " << b->sirina() << ")" << endl;

  ostr << "[ ";
  for (unsigned j=0;j<visina();j++) {
    for (unsigned i=0;i<sirina();i++) {
      ostr << redukuj_sirinu(get(i,j),sirina_broja) << " ";
    }
    if (b==0 && j<visina()-1)
      ostr << "; " << endl << "  ";

    if (b!=0) { 
      ostr << "| "; 
      for (unsigned k=0;k<(b->sirina());k++) {
        ostr << redukuj_sirinu(b->get(k,j),sirina_broja) << " ";
      }
      if ( j<visina()-1 )
        ostr << endl << "  ";
    }

  }
  ostr << "]" << endl;
  return ostr;
}

/* ispis1 daje izraz koji je kompatibilan sa programom mathematica */

ostream & matrica::ispisi1(ostream & ostr) const {
  ostr << "(" << sirina() << " x " << visina() << ")" << endl;
  ostr << "{"; 
  for (unsigned j=0;j<visina();j++) {
  ostr << "{";
    for (unsigned i=0;i<sirina();i++) {
      ostr << redukuj_sirinu(get(i,j),sirina_broja);
      if (i!=sirina()-1)
        ostr << ",";
      else 
        ostr << "}";
    }
    if (j<visina()-1)
      ostr << "," << endl << "  ";
  }
  ostr << "}" << endl;
  return ostr;
}

///////////////////////////////
// Gausovo metod eliminacije!
// slozenost algoritma: O(n^3)
// obradjeni su sledeci metodi
//    bool gaussDoleNule(matrica * M=0);      // pravi gornju trougaonu
//    bool gaussGoreNule(matrica * M=0);      // pravi donju trougaonu matr.
//    bool gaussDoleNulePivot(matrica * M=0); // gorjnja tr. sa izborom p.el.
//    bool gaussGoreNulePivot(matrica * M=0); // donja tr. sa izborom piv.el.
//    bool gaussDiag(matrica * M=0);          // dijagonalna matr.
//    bool gaussInverzna(matrica * A1);               // racuna inverznu matricu
//    bool gaussResiSistem(matrica * B, matrica * x); // resava sistem(e) jednacina
//
// pokazivac na matricu M, koji je po defaultu 0, je opcioni parametar koji
// ako se navede unosi matricu (koja mora biti iste visine kao i objekat nad 
// kojim se izvrsava metod) i iste operacije koje se izvode nad objektom izvode
// se i nad matricom M. najcesce je matrica M jedinicna, i na taj nacin se
// moze izracunati inverzna matrica


// gaussDoleNule();
// svodjenje matrice na donju trougaonu Gausovom metodom eliminacije
// bez izbora najveceg elementa!

bool matrica::gaussDoleNule(matrica * M) { 

  if (M!=0 && M->visina()!=visina())
    throw VISINE_MORAJU_BITI_JEDNAKE;
                                          // min se koristi jer matrica ne
  int i,j,min=minimum(sirina(),visina()); // mora biti kvadratna 
  for (i=0;i<min;i++) {                   // iteriranje se vrsi po dijagonali
    T a=get(i,i);                         // uzima dijagonalni element
    if (a==0) {                           // ukoliko je on nula probaj da 
      T t,tmax(0);                        // pronadjes elemnt iz te kolone
      unsigned indeks(0);                 // koji je razlicit od nule, i
      for (j=i+1;j<visina();j++) {        // izaberi najveci od takvih 
        t=get(i,j);
        if (fabs(t)>tmax) {
          tmax=t;
          indeks=j;
        }
      }
      if (index==0)                       // ukoliko nije nadjen element koji
        return false;                     // je razlicit od nule, algoritam

      if (M!=0)
        M->zameniRedove(i,indeks);

      zameniRedove(i,indeks);             // se prekida, u suprotnom se vrsi 
                                          // zamena
      a=get(i,i);                         // "a" sada uzima novu vrednost 
    }                                     // elementa sa dijagonale. tj 
    a=-1/a;                               // potrebno je -1/a

    for (j=i+1;j<visina();j++) {          // iteriraj po svim kolonama ispod
      if (get(i,j)!=0) {

        if (M!=0)
          M->pomnoziIDodajRed(i,get(i,j)*a,j);

        pomnoziIDodajRed(i,get(i,j)*a,j);   // dijagonale i eliminisi elemente
        set(i,j,0);                         // nula se postavlja zbog pogresnog 
      }
    }                                     // zaokruzivanja u okolini 0
  }
  return true;
}

// gaussGoreNule();
// svodjenje matrice na gornju trougaonu Gausovom metodom eliminacije
// bez izbora najveceg elementa!
// algoritam je slican prethodnom, uz nesto komplikovanije odredjivanje
// indeksa (posto matrica ne mora biti kvadratna). Iteracija se pocinje
// od donjeg desnog elementa, pa do sirine-minimuma, gde je minimum manja
// stranica matrice. 

bool matrica::gaussGoreNule(matrica * M) {

  if (M!=0 && M->visina()!=visina())
    throw VISINE_MORAJU_BITI_JEDNAKE;

  int i,i1,j,imin,min=minimum(sirina(),visina());
  imin=sirina()-min;
  for (i=sirina()-1,i1=visina()-1;i>=imin;i--,i1--) {
    T a=get(i,i1);                        // izbor elementa sa dijagonale

    if (a==0) {                           // ako je izabrani element nula
      T t,tmax(0);                        // probaj da nadjes neki element
      unsigned indeks(0);                 // iznad njega razlicit od nule
      for (j=i-1;j>=0;j--) {              // i to nadji najveci takav element
        t=get(i,j);
        if (fabs(t)>tmax) {
          tmax=t;
          indeks=j;
        }
      }
      if (index==0)                       // ukoliko takav element nije nadjen
        return false;                     // prekini algoritam a u suprotnom

      if (M!=0)
        M->zameniRedove(i,indeks);

      zameniRedove(i,indeks);             // zameni ta dva reda
      a=get(i,i);                         // a ponovo uzima vrednost elementa
    }                                     // na dijagonali

    a=-1/a;
    for (j=i1-1;j>=0;j--) {
      if (get(i,j)!=0) {

        if (M!=0)
          M->pomnoziIDodajRed(i,get(i,j)*a,j);

        pomnoziIDodajRed(i1,get(i,j)*a,j);
        set(i,j,0);
      }
    }

  }
  return true;
}

// gaussDoleNulePivot(short *sgn=0);
// svodjenje matrice na donju trougaonu Gausovom metodom eliminacije
// sa izborom najveceg elementa!
// Kako bi se smanjila greska racuna koja nastaje deljenjem sa elementom
// koji je nadjen na dijagonali bira se da taj element bude sto je veci
// moguc, jer je deljenje malim brojem numericki nestabilno
//
// *sgn je pokazivac na promenljivoj u kojoj ce biti smesten znak.. 
// sgn se koristi ukoliko se racuna determinanta, pri menjanju dva
// reda znak determinante se menja, te je potrebno znati sgn odnosno
// broj izmena. ako se kao *sgn navede 0 tada se ta promenljiva ne
// koristi!

bool matrica::gaussDoleNulePivot(matrica * M, short * sgn) {

  if (M!=0 && M->visina()!=visina())
    throw VISINE_MORAJU_BITI_JEDNAKE;

  int i,j,min=minimum(sirina(),visina());

  short znak=1;

  for (i=0;i<min;i++) {
    T a=get(i,i);                         // bira se element sa dijagonale
    {                                     // i pretraziuju se svi elementi 
      T t,tmax(0);                        // ispod njega kako bi se eventualno
      unsigned indeks(0);                 // pronasao neki veci od njega
      for (j=i;j<visina();j++) {          // po apsolutnoj vrednosti
        t=get(i,j);
        if (fabs(t)>tmax) {
          tmax=t;
          indeks=j;
        }
      }
      zameniRedove(i,indeks);             // vrsi se zamena
      if (i!=indeks)
        znak*=-1;

      if (M!=0)
        M->zameniRedove(i,indeks);

      a=get(i,i);
    }

    if (a==0)                             // ukoliko je posle zamene element a=0
      return false;                       // to znaci da je i pre zamene bio 0
                                          // pa se algoritam prekida

    a=-1/a;

    for (j=i+1;j<visina();j++) {
      if (get(i,j)!=0) {

        if (M!=0)
          M->pomnoziIDodajRed(i,get(i,j)*a,j);

        pomnoziIDodajRed(i,get(i,j)*a,j);
        set(i,j,0);
      }
    }
  }

  if (sgn!=0)
    *sgn=znak;
  return true;
}

// gaussDoleNulePivot();
// svodjenje matrice na gornju trougaonu Gausovom metodom eliminacije
// sa izborom najveceg elementa!
// Kako bi se smanjila greska racuna koja nastaje deljenjem sa elementom
// koji je nadjen na dijagonali bira se da taj element bude sto je veci
// moguc, jer je deljenje malim brojem numericki nestabilno


bool matrica::gaussGoreNulePivot(matrica * M) {

  if (M!=0 && M->visina()!=visina())
    throw VISINE_MORAJU_BITI_JEDNAKE;

  int i,i1,j,imin,min=minimum(sirina(),visina());
  imin=sirina()-min;
  for (i=sirina()-1,i1=visina()-1;i>=imin;i--,i1--) {
    T a=get(i,i1);

    {
      T t,tmax(0);
      unsigned indeks(0);
      for (j=i-1;j>=0;j--) {
        t=get(i,j);
        if (fabs(t)>tmax) {
          tmax=t;
          indeks=j;
        }
      }
      zameniRedove(i,indeks);

      if (M!=0)
        M->zameniRedove(i,indeks);

      a=get(i,i);
    }
    if (a==0)
      return false;

    a=-1/a;

    for (j=i1-1;j>=0;j--) {
      if (get(i,j)!=0) {

        if (M!=0)
          M->pomnoziIDodajRed(i,get(i,j)*a,j);

        pomnoziIDodajRed(i1,get(i,j)*a,j);
        set(i,j,0);
      }
    }

  }
  return true;
}

// gaussDiag()
// svodjenje matrice na dijagonalnu
// matrica se svodi prvo na donju - metodom sa izborom najveceg elementa
// a zatim na gornju (klasicnim gausovim postupkom)

bool matrica::gaussDiag(matrica * M) {

  if (M!=0 && M->visina()!=visina())
    throw VISINE_MORAJU_BITI_JEDNAKE;

  bool r1=false,r2=false;
  if (!kvadratna())
    throw MATRICA_MORA_BITI_KVADRATNA;

  r1 = gaussDoleNulePivot(M);

  if (r1)
    r2=gaussGoreNule(M);

  return r2;
}

bool matrica::gaussInverzna(matrica & A1) {
  if (!kvadratna())
    throw MATRICA_MORA_BITI_KVADRATNA;

  A1.napraviJedinicnu();
  bool r( gaussDiag(&A1) );

  if (r) { 
    for (unsigned i=0;i<visina();i++) {
      T a=1/get(i,i);
      A1.pomnoziRed(i,a);
    }
  }
  return r;
}

T matrica::gaussDeterminanta() {
  if (!kvadratna())
    throw MATRICA_MORA_BITI_KVADRATNA;

  short sgn;
  bool res = gaussDoleNulePivot(0,&sgn);
  if (res) {
    T proizvod(1.0);
    for(int i=sirina()-1;i>=0;i--)
      proizvod*=get(i,i);
    return sgn*proizvod;
  }
  else
    return 0;
}



T matrica::absSumaReda(unsigned r) {
  if (r>=visina())
    throw LOS_INDEKS_MATRICE;

  T sum(0.0);
  for(int i(sirina()-1);i>=0;i--)
    sum+=fabs(get(i,r));

  return sum;
}

T matrica::absSumaKolone(unsigned k) {
  if (k>=sirina())
    throw LOS_INDEKS_MATRICE;

  T sum(0.0);
  for(int i(visina()-1);i>=0;i--)
    sum+=fabs(get(k,i));

  return sum;
}

T matrica::normaRedova() {
  T maxSum(0.0);
  for (int i=visina()-1;i>=0;i--) {
    T sum( absSumaReda(i) );
    if (sum>maxSum)
      maxSum=sum;
  }
  return maxSum;
}

T matrica::normaKolona() {
  T maxSum(0.0);
  for (int i=sirina()-1;i>=0;i--) {
    T sum( absSumaKolone(i) );
    if (sum>maxSum)
      maxSum=sum;
  }
  return maxSum;
}

bool matrica::normaRedovaZaJacobi( T & norma ) {
  norma = -1;
  for (unsigned i=0; i<visina(); i++) {
cout << "i=" << i << " ";
    T sum( absSumaReda(i) );
    if ( norma < sum ) {
      if ( sum > 1 ) {
        norma = sum;
        break;
      } else if ( sum < 1 ) {
        norma = sum;
      }
    }
cout << "NORMA(" << i << ")=" << norma << endl;
  }
  if (norma==-1 || norma>1)
    return false;
  return true;
}

bool matrica::normaKolonaZaJacobi( T & norma ) {
}


bool matrica::gaussResiSistem(matrica * b, matrica * x) {
  if (b->visina()!=visina() ||
      b->visina()!=x->visina() ||
      b->sirina()!=x->sirina())
     throw DIMENZIJE_SE_NE_PODUDARAJU;
  if (!gaussDoleNulePivot(b))
    return false;

  for (int j=visina()-1;j>=0;j--) {
    for (int k=b->sirina()-1;k>=0;k--) {

      T t(b->get(k,j));
      for (int i=sirina()-1;i>=j+1;i--) {
        t-=get(i,j)*x->get(k,i);
      }
      x->set(k,j,t/get(j,j));

    }
  }
  return true;
}

void matrica::pripremiZaJacobi(matrica &b) {
  if (!kvadratna())
    throw MATRICA_MORA_BITI_KVADRATNA;

  for (unsigned j=0;j<visina();j++) {
    T a=get(j,j);
    set(j,j,0);
    pomnoziRed(j,-1/a);    			// POGLEDAJ DELJENJE NULOM!
    b.pomnoziRed(j,1/a);
  }
}

void matrica::jacobiIteracija(matrica &b, matrica &x, vector<T> & maxRazlike, bool gaussSaidel) const {

  unsigned n=x.visina();
  unsigned m=x.sirina(); // broj jednacina koji se resava

  T y[n][m];
  
  for (unsigned k=0;k<m;k++) 	// ovo ne treba za zajdela, moras da razdvojis ** !!!!!! **
    for (unsigned i=0;i<n;i++)
      y[i][k]=x.get(k,i);

  for (unsigned k=0;k<m;k++) {
    T maxRazlika(0.0);
    for (unsigned j=0;j<n;j++) { 
      T sum(0.0);
      for (unsigned i=0;i<n;i++) {
        if (gaussSaidel)
          sum+=x.get(k,i)*get(i,j);
        else
          sum+=y[i][k]*get(i,j);
      }
      sum+=b.get(k,j);
      x.set(k,j,sum);
      T razlika=fabs(fabs(sum)-fabs(y[j][k]));
      if (razlika>maxRazlika)
        maxRazlika=razlika;
    }
    maxRazlike[k]=maxRazlika;
  }
}

void matrica::izvadiKolonu(unsigned k, matrica & b) const {
  if (b.visina()!=visina() || b.sirina()!=1)
    throw DIMENZIJE_SE_NE_PODUDARAJU;
  if (k>=sirina())
    throw LOS_INDEKS_MATRICE;

  for (unsigned i=0; i<visina() ;i++)
    b.set(0,i,get(k,i));
}

void matrica::zameniKolonu(unsigned k, const matrica & b) {
  if (b.visina()!=visina() || b.sirina()!=1)
    throw DIMENZIJE_SE_NE_PODUDARAJU;
  if (k>=sirina())
    throw LOS_INDEKS_MATRICE;

  for (unsigned i=0; i<visina() ;i++)
    set(k,i,b.get(0,i));
}


// BROJANJE INSTRUKCIJA
// Metode koje se cesto pozivaju su inline, ukoliko se iskljuci opcija BROJANJE 
// u defs.h fajlu, tada se pozivanjem ovih funkcija ne dobija nista, tj one ce
// biti prevedene kao prazna f-ja, i verovatno ce ih optimizer c++ koda izbaciti
// a i ukoliko ih ne izbaci, one su inline te se ne trosi procesorsko vreme na
// pozivanje subrutine!


void resetujBrojace() {
#if defined BROJANJE
  matrica::broj_get_operacija = 0;
  matrica::broj_set_operacija = 0;
  matrica::broj_sabiranja = 0;
  matrica::broj_mnozenja = 0;
#endif
  startTTest(false);
}

void ispisiStanjeBrojaca(ostream & ostr) {
#if defined BROJANJE
  ostr << "--[ Stanje brojaca (od poslednjeg resetovanja) ]--" << endl;
  ostr << "broj pozivanja get metode   :" << matrica::broj_get_operacija << endl;
  ostr << "broj pozivanja set metode   :" << matrica::broj_set_operacija << endl;
  ostr << "broj sabiranja/oduzimanja   :" << matrica::broj_sabiranja << endl;
  ostr << "broj mnozenja/delenja       :" << matrica::broj_mnozenja << endl;
#endif
  finishTTest(true);
}


} // namespace

