`include "alu_codes.v"


/* Model ALU jedinice. Izracunava rezultat operacije kao i 
 flegove kada god je e_in ukljucen */
module alu(e_in, x, y, pc, op, r, o, s, z, c);
   input e_in;  // signal za aktivaciju kola
   input[31:0] x, y; // operandi
   input       pc;   // prethodni prenos
   input [3:0] op;   // operacija
   output [31:0] r;   // rezultat
   output s, z, o, c; // flegovi

   // Sledece reg promenljive se definisu fiktivno, jer im se
   // vrednosti dodeljuju u procesu. One nece zaista biti cuvane
   // u registrima, vec ce ih izracunavati kombinatorno kolo koje
   // racuna odgovarajuce ALU funkcije. Ove vrednosti se preko
   // bafera sa tri stanja prosledjuju na izlaze r, c i o. 
   reg [31:0] rc;
   reg 	      cc;
   reg 	      oc;
      
   assign r = e_in ? rc : 32'hzzzzzzzz; // bafer sa tri stanja za r
   assign o = e_in ? oc : 1'bz; // isto za o
   assign c = e_in ? cc : 1'bz; // isto za c
   assign s = e_in ? rc[31] : 1'bz; // sign flag je uvek najvisi bit rezultata
   assign z = e_in ? ~|rc : 1'bz;   // zero flag je 1 akko je rezultat 00000000

   // r, c i o racunamo u procesu koji se aktivira kad god se neki
   // od ulaznih signala promeni. carry fleg (c fleg) ce biti jedan
   // ukoliko postoji neoznaceno prekoracenje, kao i ako je prilikom
   // pomeranja sadrzaja ulevo ili udesno poslednji istisnuti bit
   // jedinica. overflow fleg (o fleg) ce biti jedinica ako dodje
   // do oznacenog prekoracenja pri sabiranju ili oduzimanju, kao i
   // prilikom promene znaka najmanjeg negativnog broja. Takodje,
   // o fleg se ukljucuje i ako se pomeranjem u levo ili u desno
   // promeni znak argumenta.
   always @(x or y or pc or op)
     case(op)
       `ALU_NOP1:
	 begin
	    rc = x;
	    oc = 1'b0;
	    cc = 1'b0;		 
	 end
       `ALU_ZERO:
	 begin
	    rc = 0;
	    oc = 1'b0;
	    cc = 1'b0;		 
	 end
       `ALU_ADD:
	 begin
	    {cc, rc} = x + y;
	    oc = x[31] ~^ y[31] && x[31] ^ rc[31];		 
	 end
       `ALU_ADC:
	 begin
	    {cc, rc} = x + y + pc;
	    oc = x[31] ~^ y[31] && x[31] ^ rc[31];		 		 
	 end
       `ALU_SUB:
	 begin
	    {cc, rc} = x - y;
	    oc = rc[31] ~^ y[31] && x[31] ^ y[31];	    
	 end
       `ALU_INC:
	 begin
	    {cc, rc} = x + 32'b1;
	    oc = (x == 32'h7fffffff);		 
	 end
       `ALU_DEC:
	 begin
	    {cc, rc} = x - 32'b1;
	    oc = (x == 32'h80000000);		 
	 end
       `ALU_NEG:
	 begin
	    rc = -x;
	    cc = 1'b0;
	    oc = (x == 32'h80000000);		 
	 end
       `ALU_AND:
	 begin
	    rc = x & y;
	    cc = 1'b0;
	    oc = 1'b0;		
	 end
       `ALU_OR:
	 begin
	    rc = x | y;
	    cc = 1'b0;
	    oc = 1'b0;		
	 end
       `ALU_XOR:
	 begin
	    rc = x ^ y;
	    cc = 1'b0;
	    oc = 1'b0;		
	 end
       `ALU_NOT:
	 begin
	    rc = ~x;
	    cc = 1'b0;
	    oc = 1'b0;		 
	 end
       `ALU_SHL:
	 begin
	    {cc, rc} = { 1'b0, x } << y;
	    oc = x[31] ^ rc[31];		 
	 end
       `ALU_SHR:
	 begin
	    {rc, cc} = { x, 1'b0 } >> y;
	    oc = 1'b0;		 
	 end
       `ALU_SAR:
	 begin
	    // Funkcija $signed() konvertuje u signed vrednost,
	    // tj. tumaci dati signal kao oznacenu vrednost, sto
	    // omogucava semantiku aritmetickog siftovanja u desno.
	    {rc, cc} = $signed({x, 1'b0}) >>> y;		 
	    oc = 1'b0;		 
	 end
       `ALU_NOP2:
	 begin
	    rc = y;
	    oc = 1'b0;
	    cc = 1'b0;		 
	 end
     endcase
   
   
   
endmodule // alu

   
